/******************************************************************************
 * (C) Copyright 2000 by Agilent Technologies GmbH. All rights reserved.      *
 ******************************************************************************/

/* ---------------------------------------------------------------
 * File: xct.c 
 *       Completer-Target (CT) control functions
 * -----------------------------------------------------------------*/

#include <xtypedef.h>

#include <xdynamic.h>
#include <xerrcapi.h>
#include <xiocommo.h>
#include <xregcons.h>
#include <xsession.h>
#include <xct.h>

#include <xsetting.h>
#include <xaddrmap.h>
#include <xloader.h>

/* Pointer to CT database */
#define DB_CT (bx_handlearray[handle].db->Exerciser.CompleterTarget)

/* Value of generic property */
#define GENERIC(prop) (DB_CT.GenProp[prop])

/* Value of split condition property */
#define SPLITCOND(i,prop) (DB_CT.SplitCondProp[i][prop])

/* Pointer to one line in behavior-memory */
#define BEHAVIOR(line) (DB_CT.Behavior.Mem[line])

/* Pointer to Behavior Update Info */
#define BEH_UPD (&DB_CT.Behavior.UpdateInfo)

/*
 * This stuct holds the addresses of the Control,ResourceBase 
 * and ResourceSize registers. They are available only for the 
 * BAR-decoders and the Expansion ROM decoder !! 
 */

/* memory map moved to xdyndata.c */

/* the following lines leaves running code running.... */
extern bx_memmaptype Mephisto_CT_Beh2Hw [];
#define Beh2Hw Mephisto_CT_Beh2Hw

/*****************************************************************************
 * CT Programming, reading
 ****************************************************************************/
/*---------------------------------------------------------------------------*
 * BestXCTProg
 *
 * Purpose: Program properties and memories from host to card
 *---------------------------------------------------------------------------*/
bx_errtype BestXCTProg (bx_handletype handle)
{
  BX_DECLARE_FUNCNAME("BestXCTProg [ctprog]");

  BX_TRY_VARS;

  BX_TRY_BEGIN
  {
    /* Switch to Prog-Mode (DBI clock) */
    BX_TRY_PROGRESS(BestXExerciserProgMode(handle,0 /* no soft force */));
  
    BX_TRY(BestXCTBehProg(handle));           /* Behavior      */
    BX_TRY(BestXCTGenProg(handle));           /* generic props */
    BX_TRY(BestXTConfigProg(handle));         /* Config space  */
    BX_TRY(BestXTDecoderProg(handle,NULL,0)); /* decoder props */
    BX_TRY(BestXCTSplitCondProg(handle));     /* split conds   */

    /* Switch to Run-Mode (PCI clock) */
    BX_TRY_PROGRESS(BX_E_OK);
    BX_TRY_PROGRESS(BestXExerciserRunMode(handle,0 /* no soft force */));
  }

  BX_TRY_CATCH
  {
    BX_TRY_PASSED
    {
      /* Switch to progmode successful */
      BX_TRY_PASSED
      {
        /* all functions in between successful */
        BX_TRY_PASSED
        {
          /* switch to runmode successful */
        }
        BX_TRY_FAILED
        {
          /* switch to runmode not successful */
          (void) BestXExerciserRunMode(handle, 1 /* soft force */);
        }
      }
      BX_TRY_FAILED
      {
        /* some function in between not successful */
        (void) BestXExerciserRunMode(handle, 0 /* no soft force */);
      }
    }
    BX_TRY_FAILED
    {
      /* switch to progmode not successful */
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXCTRead
 *
 * Purpose: Read properties and memories from card- to host memory 
 *---------------------------------------------------------------------------*/
bx_errtype BestXCTRead (bx_handletype handle, bx_int32 option)
{
  BX_DECLARE_FUNCNAME("BestXCTRead [ctread]");

  BX_TRY_VARS;

  BX_TRY_BEGIN
  {
    BX_TRY(BestXCTBehRead(handle,option));
    BX_TRY(BestXCTGenRead(handle));
    BX_TRY(BestXTConfigRead(handle));     /* Config space  */
    BX_TRY(BestXTDecoderRead(handle));    /* decoder props */
    BX_TRY(BestXCTSplitCondRead(handle)); /* split conds   */
  }

  BX_ERRETURN(BX_TRY_RET);
}

/********************************************************************
  CT generics ***************************************************
********************************************************************/

/*---------------------------------------------------------------------------*
 * BestXCTGenDefaultSet
 *
 * Purpose: Set all generic properties to default (on host only) 
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXCTGenDefaultSet(
  bx_handletype handle
)
{
  BX_DECLARE_FUNCNAME("BestXCTGenDefaultSet [ctgendefset]");

  BX_TRY_VARS;
  bx_int32 i;

  /* Generic info for range checking */
  const bx_generic_infotype *GenInfo;
  const bx_param_infotype *ParamInfo;

  BX_TRY_BEGIN
  {
    /* Get pointer to generic info */
    BX_TRY(BestXGenInfoGet(handle, BX_PARAM_CT_GEN, &GenInfo));

    for (i = 0; i < GenInfo->num_elem; i++)
    {
      /* Get pointer to i-th (existing) property */
      BX_TRY(BestXParamInfoGet(handle, BX_PARAM_CT_GEN,
            i, &ParamInfo, (bx_int32)BX_INDEX_SEARCH));

      /* Set it to default */
      BX_TRY(BestXCTGenSet(handle,
              ParamInfo->proptyp.ctgenprop,
              ParamInfo->defaultval));
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXCTGenGet 
 *
 * Purpose: Gets a generic property from host-memory 
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXCTGenGet(                     
  bx_handletype        handle,
  bx_ctgentype          prop,                      
  bx_int32             *val                       
)

{
  BX_DECLARE_FUNCNAME("BestXCTGenGet [ctgenget]");

  BX_TRY_VARS;

  BX_TRY_BEGIN
  {
    /* range checking */
    BX_TRY(BestXParamProbe(handle, BX_PARAM_CT_GEN, (bx_int32)prop));
    BX_TRY_FCT_PARAM_NULL_POINTER(val);
     
    switch (prop)
    {
      case BX_CTGEN_NUMBEH:

        /* BX_CTGEN_NUMBEH does not exist in HW.
           If user gets this property, he
           implicitly gets it from Group 0
           (all groups usually have same value):
        */
        BX_TRY(BestXCTGenGet(handle, BX_CTGEN_NUMBEHG0, val));
        break;

      default:
        /* get property from host DB */
        *val=(bx_int32)GENERIC(prop);
        break;
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXCTGenSet         
 *
 * Purpose: Sets a generic property (on host only) 
 *---------------------------------------------------------------------------*/
 
bx_errtype EXPORT BestXCTGenSet(          
  bx_handletype        handle,
  bx_ctgentype          prop,                      
  bx_int32             val                        
)
{
  BX_DECLARE_FUNCNAME("BestXCTGenSet [ctgenset]");

  BX_TRY_VARS;

  BX_TRY_BEGIN
  {
    /* range checking */
    BX_TRY(BestXParamCheck(handle, BX_PARAM_CT_GEN, (bx_int32)prop, val));

    switch (prop)
    {
      case BX_CTGEN_NUMBEH:

        /* BX_CTGEN_NUMBEH does not exist in HW.
           If user sets this property, he
           implicitly sets all groups:
        */
        BX_TRY(BestXCTGenSet(handle, BX_CTGEN_NUMBEHG0, val));
        BX_TRY(BestXCTGenSet(handle, BX_CTGEN_NUMBEHG1, val));
        BX_TRY(BestXCTGenSet(handle, BX_CTGEN_NUMBEHG2, val));
        BX_TRY(BestXCTGenSet(handle, BX_CTGEN_NUMBEHG3, val));
        BX_TRY(BestXCTGenSet(handle, BX_CTGEN_NUMBEHG4, val));
        break;

      default:
        /* set property in host DB */
        GENERIC(prop)=val;
        break;
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXCTGenProg
 *
 * Purpose: Writes all generic properties to card
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXCTGenProg(bx_handletype  handle)
{
  /* Updates Generics memory on card */

  BX_DECLARE_FUNCNAME("BestXCTGenProg [ctgenprog]");

  BX_TRY_VARS;

  BX_TRY_BEGIN
  {
    bx_int32 NumBeh;

    /* Switch to Prog-Mode (DBI clock) */
    BX_TRY_PROGRESS(BestXExerciserProgMode(handle,0 /* no soft force */));
    
    /* Number of behavior lines */
    /* Remark: BX_CTGEN_NUMBEH does not exist in HW */
    BX_TRY(BestXCTGenGet(handle,BX_CTGEN_NUMBEHG0,&NumBeh));
    BX_TRY(BestXDirectRegWrite(handle,BX_REG_TBEHAV_CTR0_LOOP_REG,sizeof(bx_int16),NumBeh-1));
    BX_TRY(BestXCTGenGet(handle,BX_CTGEN_NUMBEHG1,&NumBeh));
    BX_TRY(BestXDirectRegWrite(handle,BX_REG_TBEHAV_CTR1_LOOP_REG,sizeof(bx_int16),NumBeh-1));
    BX_TRY(BestXCTGenGet(handle,BX_CTGEN_NUMBEHG2,&NumBeh));
    BX_TRY(BestXDirectRegWrite(handle,BX_REG_TBEHAV_CTR2_LOOP_REG,sizeof(bx_int16),NumBeh-1));
    BX_TRY(BestXCTGenGet(handle,BX_CTGEN_NUMBEHG3,&NumBeh));
    BX_TRY(BestXDirectRegWrite(handle,BX_REG_TBEHAV_CTR3_LOOP_REG,sizeof(bx_int16),NumBeh-1));
    BX_TRY(BestXCTGenGet(handle,BX_CTGEN_NUMBEHG4,&NumBeh));
    BX_TRY(BestXDirectRegWrite(handle,BX_REG_TBEHAV_CTR4_LOOP_REG,sizeof(bx_int16),NumBeh-1));

    /* Switch to Run-Mode (PCI clock) */
    BX_TRY_PROGRESS(BX_E_OK);
    BX_TRY_PROGRESS(BestXExerciserRunMode(handle,0 /* no soft force */));
  }

  BX_TRY_CATCH
  {
    BX_TRY_PASSED
    {
      /* Switch to progmode successful */
      BX_TRY_PASSED
      {
        /* all functions in between successful */
        BX_TRY_PASSED
        {
          /* switch to runmode successful */
        }
        BX_TRY_FAILED
        {
          /* switch to runmode not successful */
          (void) BestXExerciserRunMode(handle, 1 /* soft force */);
        }
      }
      BX_TRY_FAILED
      {
        /* some function in between not successful */
        (void) BestXExerciserRunMode(handle, 0 /* no soft force */);
      }
    }
    BX_TRY_FAILED
    {
      /* switch to progmode not successful */
    }
  }

  
  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXCTGenRead
 *
 * Purpose: Reads all generic properties from card- to host-memory
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXCTGenRead(bx_handletype  handle)
{
  /* Updates generics memory on host DB */
  bx_int32 value=0;

  BX_DECLARE_FUNCNAME("BestXCTGenRead [ctgenread]");

  BX_TRY_VARS;

  BX_TRY_BEGIN
  {
    /* Number of behaviors */
    BX_TRY(BestXDirectRegRead(handle, BX_REG_TBEHAV_CTR0_LOOP_REG, sizeof(bx_int16), &value));
    BX_TRY(BestXCTGenSet (handle, BX_CTGEN_NUMBEHG0, value+1));

    /* NUMBEH does not exist in HW, we take group 0 (arbitrary) for it */
    BX_TRY(BestXCTGenSet (handle, BX_CTGEN_NUMBEH, value+1));

    BX_TRY(BestXDirectRegRead(handle, BX_REG_TBEHAV_CTR1_LOOP_REG, sizeof(bx_int16), &value));
    BX_TRY(BestXCTGenSet (handle, BX_CTGEN_NUMBEHG1, value+1));

    BX_TRY(BestXDirectRegRead(handle, BX_REG_TBEHAV_CTR2_LOOP_REG, sizeof(bx_int16), &value));
    BX_TRY(BestXCTGenSet (handle, BX_CTGEN_NUMBEHG2, value+1));

    BX_TRY(BestXDirectRegRead(handle, BX_REG_TBEHAV_CTR3_LOOP_REG, sizeof(bx_int16), &value));
    BX_TRY(BestXCTGenSet (handle, BX_CTGEN_NUMBEHG3, value+1));

    BX_TRY(BestXDirectRegRead(handle, BX_REG_TBEHAV_CTR4_LOOP_REG, sizeof(bx_int16), &value));
    BX_TRY(BestXCTGenSet (handle, BX_CTGEN_NUMBEHG4, value+1));
  }
  
  BX_ERRETURN(BX_TRY_RET);
}

/* --------------------------------------------------------------------------
 * Target Behavior Functions
 * ------------------------------------------------------------------------- */

/*---------------------------------------------------------------------------*
 * BestXCTBehMemInit
 *
 * Purpose: Defaults the complete behavior memory (on host only)
 * Caution: A following ExerciserProg() will re-program whole memory !!
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXCTBehMemInit(
    bx_handletype handle)
{
  BX_DECLARE_FUNCNAME ("BestXCTBehMemInit [ctbehmeminit]");

  BX_TRY_VARS;
  bx_int32 offset;
  
  BX_TRY_BEGIN
  {
    for (offset=0;offset<BX_CTBEH_MEMDEPTH;offset++)
    {
      BX_TRY(BestXCTBehDefaultSet(handle,offset));
    }
  }
  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXCTBehDefaultSet
 *
 * Purpose: Defaults one behavior line (on host)
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXCTBehDefaultSet(
  bx_handletype handle,
  bx_int32 offset)
{
  BX_DECLARE_FUNCNAME("BestXCTBehDefaultSet [ctbehdefset]");

  BX_TRY_VARS;
  bx_int32 i;

  /* Generic info for range checking */
  const bx_generic_infotype *GenInfo;
  const bx_param_infotype *ParamInfo;

  BX_TRY_BEGIN
  {
    /* Get pointer to generic info */
    BX_TRY(BestXGenInfoGet(handle, BX_PARAM_CT_BEH, &GenInfo));

    for (i = 0; i < GenInfo->num_elem; i++)
    {
      /* Get pointer to i-th (existing) property */
      BX_TRY(BestXParamInfoGet(handle, BX_PARAM_CT_BEH,
            i, &ParamInfo, (bx_int32)BX_INDEX_SEARCH));

      /* Set it to default */
      BX_TRY(BestXCTBehSet(handle,offset,
              ParamInfo->proptyp.rtbehprop,
              ParamInfo->defaultval));
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXCTBehGet
 *
 * Purpose: Gets a behavior property (from host memory)
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXCTBehGet(
  bx_handletype handle,
  bx_int32 offset,
  bx_ctbehtype prop,
  bx_int32 * val)
{
  BX_DECLARE_FUNCNAME("BestXCTBehGet [ctbehget]");

  bx_int32 i;
  bx_int32 ival;  /* partial bits of property value */

  BX_TRY_VARS;

  BX_TRY_BEGIN
  {
    /* Get pointer to i-th (existing) property */
    BX_TRY(BestXParamProbe(handle, BX_PARAM_CT_BEH, (bx_int32)prop));
    BX_TRY_FCT_PARAM_NULL_POINTER(val);

    /* Walk through all entries in table Beh2Hw[].
       There is at least one entry for each property. 
    */
    
    *val=0;
    i=0;
    do   /* Table Beh2Hw[] has at least one valid entry */
    {
      if (Beh2Hw[i].PropId==(bx_int32)prop)
      {
        /* This is an entry for property prop; we get corresponding bits */       
        ival=0;
        if (!BestXMemPropGet
        (
          (bx_int32*)(BEHAVIOR(offset)+Beh2Hw[i].MemBitPos/16),    /* pointer memory location to change */
          Beh2Hw[i].MemBitPos%16,                     /* first bitpos to read */           
          Beh2Hw[i].MemBitPos%16+Beh2Hw[i].PropLen-1, /* last bitpos to read */
          &ival,                                      /* returned bits */
          Beh2Hw[i].PropName,                         /* property name for debugging */
          sizeof(bx_int16)
          ))
        {
          
          assert(0); 
          BX_ERRETURN(BX_E_INVALID_CASE);/* fatal parameter error, should never happen */
        }

        /* merge read bits of ival into *val */
        *val = *val | ival<<Beh2Hw[i].PropBitPos; /* set bits */
      }
    } while (Beh2Hw[++i].PropId!=MEMMAP_LASTENTRY);
  }

  /* We need some modifications to adapt sw to hw */
  switch (prop)    
  {
    case BX_CTBEH_REPEAT:
      if (*val==0)
      {
        /* This register is 16 bit wide in HW.
           A value of zero in HW means 65536 in CAPI.
        */
        *val=0x10000UL;
      }
      break;
    
    case BX_CTBEH_LATENCY:
    case BX_CTBEH_SPLITLATENCY:
        /* User gets at least a value of three here.
           A three here in CAPI is represented with a zero
           in HW. So we have to increment by three.
        */ 
        *val += 3;         /* hw automatically adds 3 */
      break;

    default:
      break;
  }
  
  BX_ERRETURN(BX_E_OK);
}

/*---------------------------------------------------------------------------*
 * BestXCTBehSet
 *
 * Purpose: Sets a behavior property (host memory)
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXCTBehSet(
  bx_handletype handle,
  bx_int32 offset,
  bx_ctbehtype prop,
  bx_int32 val
)

{
  BX_DECLARE_FUNCNAME("BestXCTBehSet [ctbehset]");
  bx_int32 i;

  BX_TRY_VARS;

  BX_TRY_BEGIN
  {
    /* dynamic capability checking concerning values of parameters */
    BX_TRY(BestXParamCheck(handle, BX_PARAM_CT_BEH, (bx_int32)prop, val));

    /* We need some modifications to adapt sw to hw */
    switch (prop)        
    {
      case BX_CTBEH_REPEAT:
        if (val==0x10000UL)
        {
          /* This register is 16 bit wide in HW.
             A value of zero in HW means 65536 in CAPI
          */
      val=0UL;
        }
        break;
      
      case BX_CTBEH_LATENCY:
      case BX_CTBEH_SPLITLATENCY:
        /* User programs at least a value of three here.
           A three here in CAPI is represented with a zero
           in HW. So we have to decrement by three.
        */ 
        assert (val >= 3);   /* granted by dyncaps */  
        val -= 3;         /* hw automatically adds 3 */
        break;
      
      default:
        break;
    }

    /* Walk through all entries in table Beh2Hw[].
       There is at least one entry for each property. 
    */

    i=0;
    do   /* Table Beh2Hw[] has at least one valid entry */
    {
      if (Beh2Hw[i].PropId==(bx_int32)prop)
      {
        /* This is an entry for property prop; we set corresponding bits */       
 
        if (BestXMemPropSet
        (
          (bx_int32*)(BEHAVIOR(offset)+Beh2Hw[i].MemBitPos/16), /* pointer memory location to change */
          Beh2Hw[i].MemBitPos%16,               /* first bitpos to change */           
          Beh2Hw[i].MemBitPos%16+Beh2Hw[i].PropLen-1, /* last bitpos to change */
          val>>Beh2Hw[i].PropBitPos, /* relevant bits to set start at position 0 */
          Beh2Hw[i].PropName,        /* property name for debugging */
          sizeof(bx_int16)
        ))
        {
          /* success */
          /* mark this property modified */
          SetUpdateInfo(BEH_UPD,1,offset,Beh2Hw[i].MemBitPos/16);  
        }
        else
        {
          assert(0); 
          BX_ERRETURN(BX_E_INVALID_CASE); /* fatal parameter error, should never happen */
        }
      }
    } while (Beh2Hw[++i].PropId!=MEMMAP_LASTENTRY);
  }
  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXCTBehProg
 *
 * Purpose: Writes changes of behavior memory from host to card
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXCTBehProg(bx_handletype  handle)
{
  /* Updates CT behavior on card */

  bx_int32 decspeed=0;
  bx_int32 busspeed=0;

  BX_DECLARE_FUNCNAME("BestXCTBehProg [ctbehprog]");

  BX_TRY_VARS;

  bx_bool  Touched;  /* Wether memory needs an update at all */
  bx_int32 FirstRow,LastRow,FirstCol,LastCol; /* Range to update */

  bx_int32 i,row,col;  /* Indices */

  bx_int16ptr data=NULL;    /* part of behavior memory to be written to card */
  bx_int32 NumWords;   /* size of data[] in words */

  /* Retrieve info about the range to update */
  GetUpdateInfo(BEH_UPD,&Touched,&FirstRow,&LastRow,&FirstCol,&LastCol);
  
  BX_TRY_BEGIN
  {
    if (!Touched)
    {
      /* host and card memory are identical, no update needed */
    }
    else
    {
      /* Host- and card memory differ */

      /* Decode speed A does not work in E2930B at all, i.e. bit 0 of BX_REG_GEN_DEC_PROP_REG 
         must never be set (else ACK64# may erronously be asserted).
       */
      if (bx_handlearray[handle].hwinfo.hw!=BX_HW_E2930B)
      {
        /* Decode speed A works only, if busfrequency is below 67 MHz */
        if (FirstCol <= 4  && 4 <= LastCol) /* decspeed is in memory 4 (see xdyndata.c) */
        {
          /* Decoderspeed will be programmed in some row */
          for (row=FirstRow;row<=LastRow;row++)
          {
            BX_TRY(BestXCTBehGet(handle,row,BX_CTBEH_DECSPEED,&decspeed));
            if (decspeed==BX_CTBEH_DECSPEED_A)
            {
              /* check current frequency */
              BX_TRY(BestXStatusRead(handle,BX_STAT_BUSSPEED,&busspeed));
              if(busspeed > BX_MAXBUSSPEED_FOR_DECSPEED_A)
              {
                /* Frequency is too high, enable automatic HW switching to decode speed B (HW default) */
                BX_TRY(BestXDirectRegMaskWrite(handle,BX_REG_GEN_DEC_PROP_REG,sizeof(bx_int16),1,0));
              }
              else
              {
                /* Frequency is low enough; enable decode speed A in HW */
                BX_TRY(BestXDirectRegMaskWrite(handle,BX_REG_GEN_DEC_PROP_REG,sizeof(bx_int16),1,1));
              }
              break;  /* Skip following rows */
            }
          }
        }
      }      
      /* Switch to Prog-Mode (DBI clock) */
      BX_TRY_PROGRESS(BestXExerciserProgMode(handle,0 /* no soft force */));

      /* Min and Max registers determine the Column (property) -range 
         to program 
      */
      assert(FirstCol<BX_CTBEH_MEMWIDTH);
      assert(LastCol<BX_CTBEH_MEMWIDTH);
      BX_TRY(BestXDirectRegWrite(handle,BX_REG_TBEHAV_MIN_REG,sizeof(bx_int16),FirstCol));
      BX_TRY(BestXDirectRegWrite(handle,BX_REG_TBEHAV_MAX_REG,sizeof(bx_int16),LastCol));

      /* Set starting row (i.e. offset) from which to start programming */
      assert(FirstRow<BX_CTBEH_MEMDEPTH);
      BX_TRY(BestXDirectRegWrite(handle,BX_REG_TBEHAV_CTR,sizeof(bx_int16),FirstRow));

      /* In order to use a single call to BestXDirectRegBlockWrite() 
         instead of many calls to BestXDirectRegWrite(),
         we first have to copy the needed behavior properties into 
         a local data array 
      */
      
      NumWords=(LastRow-FirstRow+1) * (LastCol-FirstCol+1);
      data=(bx_int16ptr) BestXMemCalloc(NumWords,sizeof(bx_int16));
      if (data==NULL)
      {
        BX_ERRETURN(BX_E_HOST_MEM_FULL);
      }

      /* Fill data[] with needed properties */
      i=0;
      for (row=FirstRow;row<=LastRow;row++)
      {
        for (col=FirstCol;col<=LastCol;col++)
        {
          data[i++]=DB_CT.Behavior.Mem[row][col];
        }
      }

      assert(i==NumWords);

      /* Write to CT behavior memory */
      BX_TRY(BestXDirectRegBlockWrite(handle,
                                      BX_REG_TBEHAV_DATA_REG,
                      sizeof(bx_int16),
                                      1, /* 1=autoincrement */
                                      (bx_int8ptr) data,
                                      sizeof(bx_int16),
                                      NumWords*sizeof(bx_int16)
                                      ));

      BestXMemFree((void**) &data);

      /* Mark host memory to be identical with HW */
      SetUpdateInfo(BEH_UPD,0,0,0);

      /* Switch to Run-Mode (PCI clock) */
      BX_TRY_PROGRESS(BX_E_OK);
      BX_TRY_PROGRESS(BestXExerciserRunMode(handle,0 /* no soft force */));
    }
  }

  BX_TRY_CATCH
  {
    BX_TRY_PASSED
    {
      /* Switch to progmode successful */
      BX_TRY_PASSED
      {
        /* all functions in between successful */
        BX_TRY_PASSED
        {
          /* switch to runmode successful */
        }
        BX_TRY_FAILED
        {
          /* switch to runmode not successful */
          (void) BestXExerciserRunMode(handle, 1 /* soft force */);
        }
      }
      BX_TRY_FAILED
      {
        /* some function in between not successful */
        if (data)
        {
          BestXMemFree((void**) &data);
        }
        (void) BestXExerciserRunMode(handle, 0 /* no soft force */);
      }
    }
    BX_TRY_FAILED
    {
      /* switch to progmode not successful */
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXCTBehRead
 *
 * Purpose: Reads behavior memory from card to host
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXCTBehRead(bx_handletype  handle,bx_int32 option)
{
  /* Updates behavior memory on host DB */

  BX_DECLARE_FUNCNAME("BestXCTBehRead [ctbehread]");

  BX_TRY_VARS;

  bx_int32 NumLines=0;   /* number of lines to read */

  BX_TRY_BEGIN
  {
    /* Switch to Prog-Mode (DBI clock) */
    BX_TRY_PROGRESS(BestXExerciserProgMode(handle,0 /* no soft force */));
  
    /* Min and Max registers determine the Column (property) -range 
       to read from
    */
    BX_TRY(BestXDirectRegWrite(handle,BX_REG_TBEHAV_MIN_REG,sizeof(bx_int16),0));
    BX_TRY(BestXDirectRegWrite(handle,BX_REG_TBEHAV_MAX_REG,sizeof(bx_int16),BX_CTBEH_MEMWIDTH-1));

    /* Set starting row (i.e. offset) from which to start reading */
    BX_TRY(BestXDirectRegWrite(handle,BX_REG_TBEHAV_CTR,sizeof(bx_int16),0));

    /* Determine number of lines to read */
    switch (option)
    {
      case BX_EXEREAD_CARD:
        /* 
           The number of lines to read is determined 
           by current HW values.
        */    
        BX_TRY(BestXDirectRegRead(handle, BX_REG_TBEHAV_CTR0_LOOP_REG,sizeof(bx_int16),&NumLines));
        NumLines++; /* HW value is off by one */
        break;
      case BX_EXEREAD_ALL:
        /* Read complete memory */
        NumLines=BX_CTBEH_MEMDEPTH;
        break;
      case BX_EXEREAD_NONE:
        /* Read nothing */
        NumLines=0;
        break;
      case BX_EXEREAD_USER:
        /* 
           User can determine how many lines are read from card 
           by setting corresponding generic property before.
           Remark:
           During BestXExerciserRead(), the ..GenRead() 
           function must be called after this function,
           because it would overwrites the property with HW-values !!!
         */
        BX_TRY(BestXCTGenGet(handle, BX_CTGEN_NUMBEH, &NumLines));
        break;
      default:
        BX_E_ERROR_MSG_SET("BestXExerciserRead: Invalid value for parameter option");
        BX_TRY_ERROR(BX_E_ERROR);
        break;
    }
    
    /* Read behavior memory from card */
    BX_TRY(BestXDirectRegBlockRead(handle,
                                    BX_REG_TBEHAV_DATA_REG,
                    sizeof(bx_int16),
                                    1, /* 1=autoincrement */
                                    (bx_int8ptr)DB_CT.Behavior.Mem,
                                    sizeof(bx_int16),
                                    NumLines*BX_CTBEH_MEMWIDTH*sizeof(bx_int16)
                                    ));


    /* Switch to Run-Mode (PCI clock) */
    BX_TRY_PROGRESS(BX_E_OK);
    BX_TRY_PROGRESS(BestXExerciserRunMode(handle,0 /* no soft force */));
  }

  BX_TRY_CATCH
  {
    BX_TRY_PASSED
    {
      /* Switch to progmode successful */
      BX_TRY_PASSED
      {
        /* all functions in between successful */
        BX_TRY_PASSED
        {
          /* switch to runmode successful */
        }
        BX_TRY_FAILED
        {
          /* switch to runmode not successful */
          (void) BestXExerciserRunMode(handle, 1 /* soft force */);
        }
      }
      BX_TRY_FAILED
      {
        /* some function in between not successful */
        (void) BestXExerciserRunMode(handle, 0 /* no soft force */);
      }
    }
    BX_TRY_FAILED
    {
      /* switch to progmode not successful */
    }
  }



  /* chris: 
     Do not change the update info here, because BestXOpen()
     marks the whole memory valid (except first line).
     SetUpdateInfo(BEH_UPD,0,0,0);
  */

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXCTBehCopyLine
 *
 * Purpose: Copies one behavior line to another.
 *          Used only by application.
 *          Do not document.
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXCTBehCopyLine(
  bx_handletype  handle,
  bx_int32 srcline,
  bx_int32 destline
)
{
  BX_DECLARE_FUNCNAME("BestXCTBehCopyLine [ctbehcopyline]");
  
  bx_int32 col;
 
  BX_TRY_VARS;

  BX_TRY_BEGIN
  {
    assert(srcline<BX_CTBEH_MEMDEPTH);    
    assert(destline<BX_CTBEH_MEMDEPTH);    

    /* Set the destination line to default, i.e.
       touch it, so that it will be programmed to card
       by ExerciserProg().
    */
    BX_TRY(BestXCTBehDefaultSet(handle,destline));

    /* Copy line directly in DB */
    for (col=0;col<BX_CTBEH_MEMWIDTH;col++)
    {
      BEHAVIOR(destline)[col]=BEHAVIOR(srcline)[col];
    }
  }
 
  BX_ERRETURN(BX_TRY_RET);
}

/********************************************************************
  Target split conditions *******************************************
 *******************************************************************/

/*---------------------------------------------------------------------------*
 * BestXCTSplitCondDefaultSet
 *
 * Purpose: Sets a split condition property instance to default (on host)
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXCTSplitCondDefaultSet(
  bx_handletype handle,
  bx_int32 instance
)
{
  BX_DECLARE_FUNCNAME("BestXCTSplitCondDefaultSet [ctsplitconddefset]");

  BX_TRY_VARS;
  bx_int32 i;

  /* Generic info for range checking */
  const bx_generic_infotype *GenInfo;
  const bx_param_infotype *ParamInfo;

  BX_TRY_BEGIN
  {
    if (instance >= BX_CTSPLITPROP_DEPTH)
    {
      BX_ERRETURN(BX_E_PARAM);
    }

    /* Get pointer to generic info */
    BX_TRY(BestXGenInfoGet(handle, BX_PARAM_CT_SPLITCOND, &GenInfo));

    for (i = 0; i < GenInfo->num_elem; i++)
    {
      /* Get pointer to i-th (existing) property */
      BX_TRY(BestXParamInfoGet(handle, BX_PARAM_CT_SPLITCOND,
            i, &ParamInfo, (bx_int32)BX_INDEX_SEARCH));

      /* Set it to default */
      BX_TRY(BestXCTSplitCondSet(handle,instance,
              ParamInfo->proptyp.splitcondprop,
              ParamInfo->defaultval));
    }

    /* Decoders is on now (from dyncaps), i.e. it splits always.
       Now we set some resonable defaults.
       Decoders 0-2 bound to BAR01,23,45 resp.; decoder 3 off (compare target decode window).
       Rem: Properties for a decoders are ANDed,
            so each property must be 'true' to make the decoder respond with split. 
     */
    BX_TRY(BestXCTSplitCondSet(handle,instance,BX_CTSPLIT_DEC,(instance==3?BX_CTSPLIT_DEC_NONE:(BX_CTSPLIT_DEC_1<<(instance*2)))));    /* Used by GUI */
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXCTSplitCondSet
 *
 * Purpose: Sets a split condition property on host
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXCTSplitCondSet(
  bx_handletype handle,
  bx_int32 instance,
  bx_ctsplittype prop,                                     /* @prop */
  bx_int32 val                                          /* @val */
)
{
  BX_DECLARE_FUNCNAME("BestXCTSplitCondSet [ctsplitcondset]");

  BX_TRY_VARS;

  BX_TRY_BEGIN
  {
    /* range checking */
    BX_TRY(BestXParamCheck(handle, BX_PARAM_CT_SPLITCOND, (bx_int32)prop, val));
    if (instance >= BX_CTSPLITPROP_DEPTH)
    {
      BX_ERRETURN(BX_E_PARAM);
    }
     
    /* set property */
    SPLITCOND(instance,prop)=val;
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXCTSplitCondGet
 *
 * Purpose: Gets a split condition property from host
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXCTSplitCondGet(               
  bx_handletype handle,
  bx_int32 instance,
  bx_ctsplittype prop,                                     /* @prop */
  bx_int32   *val                                       /* #RETURN "property value: %08lx\\h" */
)
{
  BX_DECLARE_FUNCNAME("BestXCTSplitCondGet [ctsplitcondget]");

  BX_TRY_VARS;

  BX_TRY_BEGIN
  {
    /* range checking */
    BX_TRY(BestXParamProbe(handle, BX_PARAM_CT_SPLITCOND, (bx_int32)prop));
    BX_TRY_FCT_PARAM_NULL_POINTER(val);
    if (instance >= BX_CTSPLITPROP_DEPTH)
    {
      BX_ERRETURN(BX_E_PARAM);
    }
     
    /* get property */
    *val=(bx_int32)SPLITCOND(instance,prop);
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXCTSplitCondProg
 *
 * Purpose: Programs all split conditions from host to card
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXCTSplitCondProg(
  bx_handletype handle
)
{
  BX_DECLARE_FUNCNAME("BestXCTSplitCondProg [ctsplitcondprog]");

  bx_int32 i,val;

  BX_TRY_VARS;

  BX_TRY_BEGIN
  {
    /* Switch to Prog-Mode (DBI clock) */
    BX_TRY_PROGRESS(BestXExerciserProgMode(handle,0 /* no soft force */));

    /* For each of our four split decoders */
    for (i=0;i<BX_CTSPLITPROP_DEPTH;i++)
    {
      /* Mask */
      BX_TRY(BestXCTSplitCondGet(handle,i,BX_CTSPLIT_ADDRMASK_LO,&val)); 
      BX_TRY(BestXDirectRegWrite(handle,BX_REG_TSPLIT0_ADDRMASK_LO_REG+i*BX_SPLITCOND_ADDR_OFFS,
                                 sizeof(bx_int32),val)); 

      BX_TRY(BestXCTSplitCondGet(handle,i,BX_CTSPLIT_ADDRMASK_HI,&val)); 
      BX_TRY(BestXDirectRegWrite(handle,BX_REG_TSPLIT0_ADDRMASK_HI_REG+i*BX_SPLITCOND_ADDR_OFFS,
                                 sizeof(bx_int32),val)); 

      /* Value */
      BX_TRY(BestXCTSplitCondGet(handle,i,BX_CTSPLIT_ADDRVAL_LO,&val)); 
      BX_TRY(BestXDirectRegWrite(handle,BX_REG_TSPLIT0_ADDRVAL_LO_REG+i*BX_SPLITCOND_ADDR_OFFS,
                                 sizeof(bx_int32),val)); 

      BX_TRY(BestXCTSplitCondGet(handle,i,BX_CTSPLIT_ADDRVAL_HI,&val)); 
      BX_TRY(BestXDirectRegWrite(handle,BX_REG_TSPLIT0_ADDRVAL_HI_REG+i*BX_SPLITCOND_ADDR_OFFS,
                                 sizeof(bx_int32),val)); 

      /* Commands */
      BX_TRY(BestXCTSplitCondGet(handle,i,BX_CTSPLIT_CMDS,&val)); 
      BX_TRY(BestXDirectRegWrite(handle,BX_REG_TSPLIT0_CMD_REG+i*BX_SPLITCOND_ADDR_OFFS,
                                 sizeof(bx_int16),val)); 
      /* Decoders */
      BX_TRY(BestXCTSplitCondGet(handle,i,BX_CTSPLIT_DEC,&val));
      BX_TRY(BestXDirectRegWrite(handle,BX_REG_TSPLIT0_DEC_REG+i*BX_SPLITCOND_ADDR_OFFS,
                                 sizeof(bx_int16),val)); 
      /* Queues */
      BX_TRY(BestXCTSplitCondGet(handle,i,BX_CTSPLIT_QUEUE,&val)); 
      BX_TRY(BestXDirectRegWrite(handle,BX_REG_TSPLIT0_QUEUE_REG+i*BX_SPLITCOND_ADDR_OFFS,
                                 sizeof(bx_int16),val)); 
    }


    /* Switch to Run-Mode (PCI clock) */
    BX_TRY_PROGRESS(BX_E_OK);
    BX_TRY_PROGRESS(BestXExerciserRunMode(handle,0 /* no soft force */));
  }

  BX_TRY_CATCH
  {
    BX_TRY_PASSED
    {
      /* Switch to progmode successful */
      BX_TRY_PASSED
      {
        /* all functions in between successful */
        BX_TRY_PASSED
        {
          /* switch to runmode successful */
        }
        BX_TRY_FAILED
        {
          /* switch to runmode not successful */
          (void) BestXExerciserRunMode(handle, 1 /* soft force */);
        }
      }
      BX_TRY_FAILED
      {
        /* some function in between not successful */
        (void) BestXExerciserRunMode(handle, 0 /* no soft force */);
      }
    }
    BX_TRY_FAILED
    {
      /* switch to progmode not successful */
    }
  }



  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXCTSplitCondRead
 *
 * Purpose: Reads all split conditions from card to host
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXCTSplitCondRead(
  bx_handletype handle
)
{
  BX_DECLARE_FUNCNAME("BestXCTSplitCondRead [ctsplitcondread]");

  bx_int32 i,val;

  BX_TRY_VARS;

  BX_TRY_BEGIN
  {
    /* For each of our four split decoders */
    for (i=0;i<BX_CTSPLITPROP_DEPTH;i++)
    {
      /* Mask */
      BX_TRY(BestXDirectRegRead(handle,BX_REG_TSPLIT0_ADDRMASK_LO_REG+i*BX_SPLITCOND_ADDR_OFFS,
                                 sizeof(bx_int32),&val)); 
      BX_TRY(BestXCTSplitCondSet(handle,i,BX_CTSPLIT_ADDRMASK_LO,val)); 

      BX_TRY(BestXDirectRegRead(handle,BX_REG_TSPLIT0_ADDRMASK_HI_REG+i*BX_SPLITCOND_ADDR_OFFS,
                                 sizeof(bx_int32),&val)); 
      BX_TRY(BestXCTSplitCondSet(handle,i,BX_CTSPLIT_ADDRMASK_HI,val)); 

      /* Value */
      BX_TRY(BestXDirectRegRead(handle,BX_REG_TSPLIT0_ADDRVAL_LO_REG+i*BX_SPLITCOND_ADDR_OFFS,
                                 sizeof(bx_int32),&val)); 
      BX_TRY(BestXCTSplitCondSet(handle,i,BX_CTSPLIT_ADDRVAL_LO,val)); 

      BX_TRY(BestXDirectRegRead(handle,BX_REG_TSPLIT0_ADDRVAL_HI_REG+i*BX_SPLITCOND_ADDR_OFFS,
                                 sizeof(bx_int32),&val)); 
      BX_TRY(BestXCTSplitCondSet(handle,i,BX_CTSPLIT_ADDRVAL_HI,val)); 

      /* Commands */
      BX_TRY(BestXDirectRegRead(handle,BX_REG_TSPLIT0_CMD_REG+i*BX_SPLITCOND_ADDR_OFFS,
                                 sizeof(bx_int16),&val)); 
      BX_TRY(BestXCTSplitCondSet(handle,i,BX_CTSPLIT_CMDS,val)); 

      /* Decoders */
      BX_TRY(BestXDirectRegRead(handle,BX_REG_TSPLIT0_DEC_REG+i*BX_SPLITCOND_ADDR_OFFS,
                                 sizeof(bx_int16),&val)); 
      BX_TRY(BestXCTSplitCondSet(handle,i,BX_CTSPLIT_DEC,val)); 

      /* Queues */
      BX_TRY(BestXDirectRegRead(handle,BX_REG_TSPLIT0_QUEUE_REG+i*BX_SPLITCOND_ADDR_OFFS,
                                 sizeof(bx_int16),&val)); 
      BX_TRY(BestXCTSplitCondSet(handle,i,BX_CTSPLIT_QUEUE,val)); 
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}
